import 'package:flutter/material.dart';
import 'package:feather_icons/feather_icons.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:wordpress_app/blocs/notification_bloc.dart';
import 'package:wordpress_app/blocs/settings_bloc.dart';
import 'package:wordpress_app/blocs/theme_bloc.dart';
import 'package:wordpress_app/blocs/user_bloc.dart';
import 'package:wordpress_app/config/config.dart';
import 'package:wordpress_app/config/wp_config.dart';
import 'package:wordpress_app/models/app_config_model.dart';
import 'package:wordpress_app/pages/login.dart';
import 'package:wordpress_app/pages/security.dart';
import 'package:wordpress_app/pages/welcome.dart';
import 'package:wordpress_app/services/app_service.dart';
import 'package:wordpress_app/utils/next_screen.dart';
import 'package:easy_localization/easy_localization.dart';
import '../blocs/config_bloc.dart';
import '../blocs/font_size_bloc.dart';
import '../widgets/language.dart';

class SettingPage extends ConsumerStatefulWidget {
  const SettingPage({super.key});

  @override
  ConsumerState<SettingPage> createState() => _SettingPageState();
}

class _SettingPageState extends ConsumerState<SettingPage>
    with AutomaticKeepAliveClientMixin {
  void openLicenceDialog() {
    final SettingsBloc sb = ref.read(settingsBlocProvider);
    showDialog(
        context: context,
        builder: (_) {
          return AboutDialog(
            applicationName: Config.appName,
            applicationVersion: sb.appVersion,
            applicationIcon: const Image(
              image: AssetImage(Config.appIcon),
              height: 30,
              width: 30,
            ),
          );
        });
  }

  @override
  Widget build(BuildContext context) {
    super.build(context);
    final ub = ref.watch(userBlocProvider);
    final configs = ref.read(configBlocProvider).configs!;
    return Scaffold(
        body: CustomScrollView(
      slivers: [
        SliverAppBar(
          automaticallyImplyLeading: false,
          expandedHeight: 140,
          pinned: true,
          backgroundColor: Theme.of(context).primaryColor,
          flexibleSpace: FlexibleSpaceBar(
            centerTitle: false,
            title: const Text('settings',
                    style: TextStyle(
                        fontWeight: FontWeight.w600, color: Colors.white))
                .tr(),
            titlePadding:
                const EdgeInsets.only(left: 20, bottom: 20, right: 20),
          ),
        ),
        SliverToBoxAdapter(
          child: SingleChildScrollView(
            padding: const EdgeInsets.only(top: 20, bottom: 20),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Visibility(
                  visible: configs.loginEnabled,
                  child: Container(
                      decoration:
                          BoxDecoration(color: Theme.of(context).cardColor),
                      child: !ub.isSignedIn
                          ? const GuestUserUI()
                          : const UserUI()),
                ),
                const SizedBox(
                  height: 15,
                ),
                Container(
                  margin: const EdgeInsets.symmetric(horizontal: 20),
                  decoration: BoxDecoration(
                    color: Theme.of(context).cardColor,
                    borderRadius: BorderRadius.circular(12),
                  ),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Padding(
                        padding: const EdgeInsets.only(top: 20, bottom: 10),
                        child: Text(
                          'general-settings',
                          style:
                              Theme.of(context).textTheme.titleMedium?.copyWith(
                                    fontWeight: FontWeight.w600,
                                  ),
                        ).tr(),
                      ),
                      _ModernListTile(
                        icon: FeatherIcons.bell,
                        title: 'get-notifications'.tr(),
                        isFirst: true,
                        trailing: Switch.adaptive(
                          inactiveThumbColor:
                              Theme.of(context).colorScheme.outline,
                          value: ref.watch(notificationBlocProvider).subscribed,
                          onChanged: (bool value) => ref
                              .read(notificationBlocProvider)
                              .handleSubscription(context, value),
                        ),
                        onTap: null,
                      ),
                      _ModernListTile(
                        icon: Icons.wb_sunny,
                        title: 'dark-mode'.tr(),
                        iconColor:
                            Theme.of(context).colorScheme.onSecondaryContainer,
                        trailing: Switch.adaptive(
                          inactiveThumbColor:
                              Theme.of(context).colorScheme.outline,
                          value: ref.watch(themeBlocProvider).darkTheme ?? true,
                          onChanged: (_) {
                            ref.read(themeBlocProvider).toggleTheme();
                          },
                        ),
                        onTap: null,
                      ),
                      _ModernListTile(
                        icon: Icons.text_fields,
                        title: 'font-size'.tr(),
                        iconColor:
                            Theme.of(context).colorScheme.onTertiaryContainer,
                        trailing: _FontSizeSelector(),
                        onTap: () => _showFontSizeDialog(context, ref),
                      ),
                      if (configs.multiLanguageEnabled)
                        _ModernListTile(
                          icon: Icons.translate,
                          title: 'language'.tr(),
                          iconColor:
                              Theme.of(context).colorScheme.onTertiaryContainer,
                          isLast: true,
                          onTap: () => nextScreenPopupiOS(
                              context, const LanguagePopup()),
                        ),
                    ],
                  ),
                ),
                const SizedBox(
                  height: 15,
                ),
                Container(
                  margin:
                      const EdgeInsets.symmetric(horizontal: 20, vertical: 20),
                  decoration: BoxDecoration(
                    color: Theme.of(context).cardColor,
                    borderRadius: BorderRadius.circular(12),
                  ),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Text(
                        'about-app',
                        style:
                            Theme.of(context).textTheme.titleMedium?.copyWith(
                                  fontWeight: FontWeight.w600,
                                ),
                      ).tr(),
                      const SizedBox(height: 10),
                      _ModernListTile(
                        icon: FeatherIcons.lock,
                        title: 'privacy-policy'.tr(),
                        iconColor:
                            Theme.of(context).colorScheme.onPrimaryContainer,
                        isFirst: true,
                        onTap: () => AppService().openLinkWithCustomTab(
                            context, configs.priivacyPolicyUrl, ref),
                      ),
                      _ModernListTile(
                        icon: FeatherIcons.star,
                        title: 'rate-app'.tr(),
                        iconColor:
                            Theme.of(context).colorScheme.onSecondaryContainer,
                        isLast: true,
                        onTap: () => AppService().launchAppReview(context, ref),
                      ),
                      _ModernListTile(
                        icon: FeatherIcons.mail,
                        title: 'contact-us'.tr(),
                        iconColor:
                            Theme.of(context).colorScheme.onErrorContainer,
                        isFirst: true,
                        onTap: () => AppService()
                            .openEmailSupport(context, configs.supportEmail),
                      ),
                      _ModernListTile(
                        icon: FeatherIcons.link,
                        title: 'our-website'.tr(),
                        onTap: () => AppService().openLinkWithCustomTab(
                            context, WpConfig.baseURL, ref),
                      ),
                    ],
                  ),
                ),
                const SizedBox(
                  height: 15,
                ),
                _socialIcons(configs, context),
              ],
            ),
          ),
        ),
      ],
    ));
  }

  Widget _socialIcons(ConfigModel configs, BuildContext context) {
    final List<Map<String, dynamic>> socialLinks = [];

    if (configs.fbUrl != '') {
      socialLinks.add({
        'icon': FeatherIcons.facebook,
        'url': configs.fbUrl,
        'color': Theme.of(context).colorScheme.onPrimaryContainer,
      });
    }
    if (configs.youtubeUrl != '') {
      socialLinks.add({
        'icon': FeatherIcons.youtube,
        'url': configs.youtubeUrl,
        'color': Theme.of(context).colorScheme.onErrorContainer,
      });
    }
    if (configs.instagramUrl != '') {
      socialLinks.add({
        'icon': FeatherIcons.instagram,
        'url': configs.instagramUrl,
        'color': Theme.of(context).colorScheme.onTertiaryContainer,
      });
    }
    if (configs.twitterUrl != '') {
      socialLinks.add({
        'icon': FeatherIcons.twitter,
        'url': configs.twitterUrl,
        'color': Theme.of(context).colorScheme.onPrimaryContainer,
      });
    }
    if (configs.threadsUrl != '') {
      socialLinks.add({
        'icon': FeatherIcons.atSign,
        'url': configs.threadsUrl,
        'color': Theme.of(context).colorScheme.onSurfaceVariant,
      });
    }

    if (socialLinks.isEmpty) return const SizedBox.shrink();

    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 20),
      child: Container(
        padding: const EdgeInsets.all(16),
        decoration: BoxDecoration(
          color: Theme.of(context).cardColor,
          borderRadius: BorderRadius.circular(12),
          border: Border.all(
            color: Theme.of(context).colorScheme.outline.withValues(alpha: 0.2),
            width: 1,
          ),
        ),
        child: Row(
          mainAxisAlignment: MainAxisAlignment.center,
          children: socialLinks.map((link) {
            return Padding(
              padding: const EdgeInsets.symmetric(horizontal: 8),
              child: InkWell(
                onTap: () => AppService().openLink(context, link['url']),
                borderRadius: BorderRadius.circular(10),
                child: Container(
                  width: 48,
                  height: 48,
                  decoration: BoxDecoration(
                    border: Border.all(
                      color: Theme.of(context)
                          .colorScheme
                          .outline
                          .withValues(alpha: 0.2),
                      width: 1,
                    ),
                    borderRadius: BorderRadius.circular(10),
                  ),
                  child: Icon(
                    link['icon'],
                    size: 22,
                    color: link['color'],
                  ),
                ),
              ),
            );
          }).toList(),
        ),
      ),
    );
  }

  @override
  bool get wantKeepAlive => true;
}

class _ModernListTile extends StatelessWidget {
  final IconData icon;
  final String title;
  final VoidCallback? onTap;
  final Widget? trailing;
  final Color? iconColor;
  final bool isFirst;
  final bool isLast;

  const _ModernListTile({
    required this.icon,
    required this.title,
    this.onTap,
    this.trailing,
    this.iconColor,
    this.isFirst = false,
    this.isLast = false,
  });

  @override
  Widget build(BuildContext context) {
    return InkWell(
      onTap: onTap,
      borderRadius: BorderRadius.vertical(
        top: isFirst ? const Radius.circular(12) : Radius.zero,
        bottom: isLast ? const Radius.circular(12) : Radius.zero,
      ),
      child: Container(
        padding: const EdgeInsets.symmetric(horizontal: 0, vertical: 8),
        decoration: BoxDecoration(
          border: Border(
            bottom: isLast
                ? BorderSide.none
                : BorderSide(
                    color: Theme.of(context)
                        .colorScheme
                        .outline
                        .withValues(alpha: 0.1),
                    width: 0.5,
                  ),
          ),
        ),
        child: Row(
          children: [
            Container(
              width: 40,
              height: 40,
              decoration: BoxDecoration(
                color: (iconColor ?? Theme.of(context).colorScheme.primary)
                    .withValues(alpha: 0.1),
                borderRadius: BorderRadius.circular(10),
              ),
              child: Icon(
                icon,
                size: 20,
                color: iconColor ?? Theme.of(context).colorScheme.primary,
              ),
            ),
            const SizedBox(width: 16),
            Expanded(
              child: Text(
                title,
                style: Theme.of(context).textTheme.bodyLarge?.copyWith(
                      fontWeight: FontWeight.w500,
                    ),
              ),
            ),
            if (trailing != null) ...[
              const SizedBox(width: 8),
              trailing!,
            ] else if (onTap != null)
              Icon(
                FeatherIcons.chevronRight,
                size: 18,
                color: Theme.of(context).colorScheme.onSurfaceVariant,
              ),
          ],
        ),
      ),
    );
  }
}

class GuestUserUI extends StatelessWidget {
  const GuestUserUI({super.key});

  @override
  Widget build(BuildContext context) {
    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 20),
      decoration: BoxDecoration(
        color: Theme.of(context).cardColor,
        borderRadius: BorderRadius.circular(12),
      ),
      child: _ModernListTile(
        icon: FeatherIcons.userPlus,
        title: 'login'.tr(),
        iconColor: Theme.of(context).colorScheme.onPrimaryContainer,
        isFirst: true,
        isLast: true,
        onTap: () => nextScreenPopupiOS(
            context,
            const LoginPage(
              popUpScreen: true,
            )),
      ),
    );
  }
}

class UserUI extends ConsumerWidget {
  const UserUI({super.key});

  @override
  Widget build(BuildContext context, WidgetRef ref) {
    final UserBloc ub = ref.watch(userBlocProvider);
    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 20),
      decoration: BoxDecoration(
        color: Theme.of(context).cardColor,
        borderRadius: BorderRadius.circular(12),
      ),
      child: Column(
        children: [
          _ModernListTile(
            icon: FeatherIcons.userCheck,
            title: ub.name.toString(),
            iconColor: Theme.of(context).colorScheme.onPrimaryContainer,
            isFirst: true,
            onTap: null,
          ),
          _ModernListTile(
            icon: FeatherIcons.mail,
            title: ub.email.toString(),
            iconColor: Theme.of(context).colorScheme.onSecondaryContainer,
            onTap: null,
          ),
          _ModernListTile(
            icon: FeatherIcons.settings,
            title: 'security'.tr(),
            iconColor: Theme.of(context).colorScheme.onErrorContainer,
            isFirst: true,
            isLast: true,
            onTap: () => nextScreen(context, const SecurityPage()),
          ),
          _ModernListTile(
            icon: FeatherIcons.logOut,
            title: 'logout'.tr(),
            iconColor: Theme.of(context).colorScheme.onErrorContainer,
            isLast: true,
            onTap: () => openLogoutDialog(context, ref),
          ),
        ],
      ),
    );
  }

  void openLogoutDialog(BuildContext context, WidgetRef ref) {
    showDialog(
        context: context,
        builder: (BuildContext context) {
          return AlertDialog(
            content: const Text('logout-description').tr(),
            title: const Text('logout-title').tr(),
            actions: <Widget>[
              TextButton(
                  onPressed: () {
                    Navigator.pop(context);
                  },
                  child: const Text('Cancel')),
              TextButton(
                  onPressed: () async {
                    Navigator.pop(context);
                    _handleLogout(context, ref);
                  },
                  child: const Text('logout').tr()),
            ],
          );
        });
  }

  Future _handleLogout(BuildContext context, WidgetRef ref) async {
    final UserBloc ub = ref.read(userBlocProvider);
    await ub.userSignout().then((value) {
      if (!context.mounted) return;
      nextScreenCloseOthersAnimation(context, const WelcomePage());
    });
  }
}

class _FontSizeSelector extends ConsumerWidget {
  const _FontSizeSelector();

  @override
  Widget build(BuildContext context, WidgetRef ref) {
    final fontSizeBloc = ref.watch(fontSizeBlocProvider);
    
    String currentSize;
    switch (fontSizeBloc.fontSize) {
      case FontSizeOption.small:
        currentSize = 'small'.tr();
        break;
      case FontSizeOption.medium:
        currentSize = 'medium'.tr();
        break;
      case FontSizeOption.large:
        currentSize = 'large'.tr();
        break;
    }
    
    return Text(
      currentSize,
      style: Theme.of(context).textTheme.bodyMedium?.copyWith(
            color: Theme.of(context).colorScheme.onSurfaceVariant,
            fontWeight: FontWeight.w500,
          ),
    );
  }
}

void _showFontSizeDialog(BuildContext context, WidgetRef ref) {
  final fontSizeBloc = ref.read(fontSizeBlocProvider);
  
  showDialog(
    context: context,
    builder: (BuildContext context) {
      return AlertDialog(
        title: Text('font-size').tr(),
        content: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            _FontSizeOptionTile(
              label: 'small'.tr(),
              isSelected: fontSizeBloc.fontSize == FontSizeOption.small,
              onTap: () {
                ref.read(fontSizeBlocProvider).setFontSize(FontSizeOption.small);
                Navigator.pop(context);
              },
            ),
            _FontSizeOptionTile(
              label: 'medium'.tr(),
              isSelected: fontSizeBloc.fontSize == FontSizeOption.medium,
              onTap: () {
                ref.read(fontSizeBlocProvider).setFontSize(FontSizeOption.medium);
                Navigator.pop(context);
              },
            ),
            _FontSizeOptionTile(
              label: 'large'.tr(),
              isSelected: fontSizeBloc.fontSize == FontSizeOption.large,
              onTap: () {
                ref.read(fontSizeBlocProvider).setFontSize(FontSizeOption.large);
                Navigator.pop(context);
              },
            ),
          ],
        ),
      );
    },
  );
}

class _FontSizeOptionTile extends StatelessWidget {
  final String label;
  final bool isSelected;
  final VoidCallback onTap;

  const _FontSizeOptionTile({
    required this.label,
    required this.isSelected,
    required this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    return ListTile(
      title: Text(label),
      trailing: isSelected
          ? Icon(
              Icons.check_circle,
              color: Theme.of(context).colorScheme.primary,
            )
          : const Icon(Icons.circle_outlined),
      selected: isSelected,
      selectedTileColor: Theme.of(context).colorScheme.primaryContainer,
      onTap: onTap,
    );
  }
}
